define(['angular', 'app', 'lodash'], function (angular, app) {
	"use strict";

	app.service('FitnessGraphService', function ($filter, TrackersConstants) {


		var service = {};
        var getMoodPlotBands = function() {
            var c1 = 'rgba(68, 170, 213, 0.1)';
            return [
                {from: 1, to: 2,  color: c1},
                {from: 3, to: 4,  color: c1},
                {from: 5, to: 6,  color: c1},
                {from: 7, to: 8,  color: c1},
                {from: 9, to: 10, color: c1}
            ];
        };


		var getChartConfig = function(section, title) {
			var chartConfig = {
				options: {
					chart: {
						type: 'column',
						borderColor: '#EBBA95',
						borderWidth: 2,
						borderRadius: 3,
						zoomType: 'x'
					},
					plotOptions: {
						series: {
							marker: { radius: 6 },
							events: {
								legendItemClick: function () {
									return false;
								}
							}
						}
					}
				},
				useHighStocks: false,
				title: { text: title },
				credits: { enabled: false },
				loading: false,
				legend: { enabled: true }
			};

			return chartConfig;
		};

		var getDateUTC = function(date, truncateTime) {
			return Date.UTC(
				parseInt($filter('date')(date, 'yyyy')),
				parseInt($filter('date')(date, 'MM')) - 1,
				parseInt($filter('date')(date, 'dd')),
				truncateTime ? 0 : parseInt($filter('date')(date, 'HH')),
				truncateTime ? 0 : parseInt($filter('date')(date, 'mm'))
			);
		};

		var initTmpSeries = function(array) {
			var tmp = {};
			angular.forEach(array, function(item) {
				tmp[item] = [];
			});
			return tmp;
		};

		function getSeriesValue(valueQuantity, emptyValue) {



			return (valueQuantity && angular.isDefined(valueQuantity.value) && (angular.isNumber(valueQuantity.value) || valueQuantity.value.length > 0))
				? parseFloat(valueQuantity.value) : emptyValue;
		}

		var processGlucoseSeries = function(points) {
			var connectingLineText = 'connecting-line',
				seriesList = angular.copy(TrackersConstants.glucose.entry.testConditions);




			seriesList.unshift(connectingLineText);
			var tmpSeries = initTmpSeries(seriesList);


			angular.forEach(angular.copy(points).reverse(), function (point) {
				var recordedDate = new Date(point.effectiveDateTime),
					recordedDateUTC = getDateUTC(recordedDate),
					method = point.component[0].valueString,
					seriesValue = getSeriesValue(point.valueQuantity, null);

				if(tmpSeries[method] && !isNaN(seriesValue)) {
					tmpSeries[method].push({x: recordedDateUTC, y: seriesValue, method: method});
					tmpSeries[connectingLineText].push({x: recordedDateUTC, y: seriesValue, method: method});
				}
			});

			var chartSeries = [];
			for(var method in tmpSeries) {
				chartSeries.push({
					name: method,
					data: tmpSeries[method],
					showInLegend: method !== connectingLineText,
					type: method === connectingLineText ? 'spline' : 'scatter',
					marker: {radius: method === connectingLineText ? 1 : 6}
				});
			}

			return chartSeries;
		};


		var processFitnessActivitySeries =  function(points,metaData) {

			var tmpSeries = {};

			angular.forEach(angular.copy(points).reverse(), function(point) {
				var dateUTC = getDateUTC(new Date(point['effectiveDateTime'])),
					component = point.component;

				angular.forEach(component, function(series, seriesIndex){
					if (angular.isDefined(metaData[seriesIndex])) {
						var seriesName = metaData[seriesIndex].label,
							seriesValue = getSeriesValue(series.valueQuantity, null);

						if (!tmpSeries[seriesName])
							tmpSeries[seriesName] = [];
						tmpSeries[seriesName].push([dateUTC, seriesValue]);
					}
				});
			});

			var chartSeries = [];
			for(var dataType in tmpSeries) {
				chartSeries.push({
					name: dataType,
					data: tmpSeries[dataType],
					showInLegend: true,
					type: dataType === 'column'
				});
			}
			return chartSeries;
		};


		var processFitnessGoalsSeries =  function(points,metaData) {

			var tmpSeries = {};

			angular.forEach(angular.copy(points).reverse(), function(point) {
				var dateUTC = getDateUTC(new Date(point['effectiveDateTime'])),
					component = point.component;

				angular.forEach(component, function(series, seriesIndex){
					if (angular.isDefined(metaData[seriesIndex])) {
						var seriesName = metaData[seriesIndex].label,
							seriesValue = getSeriesValue(series.valueQuantity, null);

						if (!tmpSeries[seriesName])
							tmpSeries[seriesName] = [];
						tmpSeries[seriesName].push([dateUTC, seriesValue]);
					}
				});
			});

			var chartSeries = [];
			for(var dataType in tmpSeries) {
				chartSeries.push({
					name: dataType,
					data: tmpSeries[dataType],
					showInLegend: true,
					type: dataType === 'column'
				});
			}
			return chartSeries;
		};


		var processMoodSeries = function(points) {
            var moodText = 'Mood',
                tmpSeries = initTmpSeries([moodText]);

            var getMoodIcon = function(mood) {
                if(mood >= 1 && mood <=  2) return 'url(assets/images/app-icons/mood/mood-verybad-16x16.png)';
                if(mood >= 3 && mood <=  4) return 'url(assets/images/app-icons/mood/mood-bad-16x16.png)';
                if(mood >= 5 && mood <=  6) return 'url(assets/images/app-icons/mood/mood-neutral-16x16.png)';
                if(mood >= 7 && mood <=  8) return 'url(assets/images/app-icons/mood/mood-good-16x16.png)';
                if(mood >= 9 && mood <= 10) return 'url(assets/images/app-icons/mood/mood-verygood-16x16.png)';
            };

            angular.forEach(angular.copy(points).reverse(), function(point) {
                var dateUTC = getDateUTC(new Date(point['recorded'])),
                    seriesValue = getSeriesValue(point.valueQuantity, null);

                if (angular.isNumber(seriesValue)) {
                    tmpSeries[moodText].push({
                        x: dateUTC,
                        y: seriesValue,
                        marker: {symbol: getMoodIcon(seriesValue)}
                    });
                }
            });

            return [
                {name: moodText, type: 'scatter', data: tmpSeries[moodText], color: '#f7a35c'}
            ];
        };


		var processCholesterolSeries = function(points, metaData) {
			var tmpSeries = {};

			angular.forEach(angular.copy(points).reverse(), function(point) {
				var dateUTC = getDateUTC(new Date(point['effectiveDateTime'])),
					component = point.component;

				angular.forEach(component, function(series, seriesIndex){
					if (angular.isDefined(metaData[seriesIndex])) {
						var seriesName = metaData[seriesIndex].label,
							seriesValue = getSeriesValue(series.valueQuantity, null);

						if (!tmpSeries[seriesName])
							tmpSeries[seriesName] = [];
						tmpSeries[seriesName].push([dateUTC, seriesValue]);
					}
				});
			});

			var chartSeries = [];
			for(var dataType in tmpSeries) {
				chartSeries.push({
					name: dataType,
					data: tmpSeries[dataType],
					showInLegend: true,
					type: dataType === 'column'
				});
			}

			return chartSeries;
		};



		var processBPPSeries = function(points, metaData) {
			var tmpSeries = {};

			angular.forEach(angular.copy(points).reverse(), function(point) {
				var dateUTC = getDateUTC(new Date(point['effectiveDateTime'])),
					component = point.component;

				angular.forEach(component, function(series, seriesIndex){
					if (angular.isDefined(metaData[seriesIndex])) {
						var seriesName = metaData[seriesIndex].label,
							seriesValue = getSeriesValue(series.valueQuantity, null);

						if (!tmpSeries[seriesName])
							tmpSeries[seriesName] = [];
						tmpSeries[seriesName].push([dateUTC, seriesValue]);
					}
				});
			});

			var chartSeries = [];
			for(var dataType in tmpSeries) {
				chartSeries.push({
					name: dataType,
					data: tmpSeries[dataType],
					showInLegend: true,
					type: dataType === 'column'
				});
			}

			return chartSeries;
		};

		var processSingleSeries = function(section, points, metaData) {
			var chartSeries = [{
				name: metaData[0].label,
				type: 'column',
				data: []
			}];

			angular.forEach(angular.copy(points).reverse(), function(point) {
				var dateUTC = getDateUTC(new Date(point['recorded'] || point['effectiveDateTime'])),
					seriesValue = getSeriesValue(point.valueQuantity, null);

				if (angular.isNumber(seriesValue)) {
					chartSeries[0].data.push([dateUTC, seriesValue]);
				}
			});

			return chartSeries;
		};




        var getSeriesConfig = function(section, points, metaData) {
			if (!points || points.length === 0) { return []; }

			if(section === 'glucose') {
				return processGlucoseSeries(points);

            } else if (section === 'cholesterol') {
				return processCholesterolSeries(points, metaData);

			} else if(section === 'fitness-activity') {
				return processFitnessActivitySeries(points, metaData);
			} else if(section === 'fitness-goals') {
				return processFitnessGoalsSeries(points, metaData);

			} else if(section === 'bp-pulse') {
				return processBPPSeries(points, metaData);

			} else {
				return processSingleSeries(section, points, metaData);
			}

		};

		var getXaxisConfig = function (section, filter) {



			var startDate = new Date(filter.startDate),
				endDate = new Date(filter.endDate),
				startDateUTC = Date.UTC(
					parseInt($filter('date')(startDate, 'yyyy')),
					parseInt($filter('date')(startDate, 'MM')) - 1,
					parseInt($filter('date')(startDate, 'dd')),
					0, 0
				),
				endDateUTC = Date.UTC(
					parseInt($filter('date')(endDate, 'yyyy')),
					parseInt($filter('date')(endDate, 'MM')) - 1,
					parseInt($filter('date')(endDate, 'dd')) + 1,
					0, 0
				),
				title = 'Date/Time';

			return {
				type: 'datetime',
				dateTimeLabelFormats: {
					millisecond: '%b %e %I:%M %p',
					second: '%b %e %I:%M %p',
					minute: '%b %e %I:%M %p',
					hour: '%b %e %I:%M %p',
					day: '%b %e',
					week: '%b %e',
					month: '%b %Y',
					year: '%Y'
				},
				minRange: 24 * 3600000,
				minTickInterval: 3600000,
				min: startDateUTC,
				max: endDateUTC,
				title: {
					text: title
				}
			};
		};

		var getYaxisConfig = function(section, metaData) {	 
			var style = {'font-weight': 'bold', 'color': 'black', 'font-size': '12px'};

			if(section === 'bp-pulse') {
				return [
					{min: 0, tickInterval: 25, title: {text: 'Blood Pressure' + ' (' + metaData[0].units + ')', style: style}},
					{min: 0, max: 25, tickInterval: 25, title: {text: 'Pulse' + ' (' + metaData[2].units + ')', style: style}, opposite: true}
				];
			}
			return {
				min: 0, max: section === 'mood' ? 10.5 : null,
				endOnTick: section !== 'mood',
				allowDecimals: false,
				tickInterval: section === 'mood' ? 1 : 25,
				title: {text: metaData[0].label + ' (' + metaData[0].units + ')', style: style},
				plotBands: section === 'mood' ? getMoodPlotBands() : null
			};
		};

		var configureTooltip = function (section, metaData) {
			var tooltip = {
				shared: section !== "glucose",
				crosshairs: [true]
			};

			if (section === "glucose") {
				tooltip.formatter = function () {
					return Highcharts.dateFormat('%A %m/%d/%Y %I:%M %p', this.x) + '<br>'  + ' Glucose: <b>' + this.y + '</b> mg/dl (' + this.point.series.name + ')';
				};
			}
            else if (section === "mood") {
                tooltip.formatter = function () {
                    return Highcharts.dateFormat('%A %m/%d/%Y %I:%M %p', this.x) + '<br>' + metaData[0].label + ': <b>' + this.y + '</b> ' + metaData[0].units;
                };
            }

			else if (section === "weight") {
				tooltip.formatter = function () {
					return Highcharts.dateFormat('%A %m/%d/%Y %I:%M %p', this.x) + '<br>' + metaData[0].label + ': <b>' + this.y + '</b> ' + metaData[0].units;
				};
			}
			else if (section === "fitness-goals") {
				tooltip.formatter = function () {
					var extraString = '';
					_.each(this.points, function (pt) {
						var units = _.pluck(_.filter(metaData, { label: pt.series.name }), 'units'),
							coloredCircle = '<tspan style="fill:' + pt.series.color + '" x="8" dy="16">●</tspan> '; // circle acting as legend color mapping

						extraString += '<br>' + coloredCircle +  '<span>' + pt.series.name + '</span>: <b>' + pt.y + '</b> ' + units;
					});

					return Highcharts.dateFormat('%A %m/%d/%Y %I:%M %p', this.x) + extraString;
				};
			}

            else if (section === "fitness-activity") {
				tooltip.formatter = function () {
					var extraString = '';
					_.each(this.points, function (pt) {
						var units = _.pluck(_.filter(metaData, { label: pt.series.name }), 'units'),
							coloredCircle = '<tspan style="fill:' + pt.series.color + '" x="8" dy="16">●</tspan> '; // circle acting as legend color mapping

						extraString += '<br>' + coloredCircle +  '<span>' + pt.series.name + '</span>: <b>' + pt.y + '</b> ' + units;
					});

					return Highcharts.dateFormat('%A %m/%d/%Y %I:%M %p', this.x) + extraString;
				};
            }
            else if (section === "cholesterol") {
                tooltip.formatter = function () {
                    return Highcharts.dateFormat('%A %m/%d/%Y %I:%M %p', this.x) + '<br>' + metaData[0].label + ': <b>' + this.y + '</b> ' + metaData[0].units;
                };
            }
            else if (section === 'bp-pulse') {
				tooltip.formatter = function () {
					var extraString = '';
					_.each(this.points, function (pt) {
						var units = _.pluck(_.filter(metaData, { label: pt.series.name }), 'units'),
							coloredCircle = '<tspan style="fill:' + pt.series.color + '" x="8" dy="16">●</tspan> '; // circle acting as legend color mapping

						extraString += '<br>' + coloredCircle +  '<span>' + pt.series.name + '</span>: <b>' + pt.y + '</b> ' + units;
					});

					return Highcharts.dateFormat('%A %m/%d/%Y %I:%M %p', this.x) + extraString;
				};
			}
			return tooltip;
		};

		service.getGraphConfig = function(section, filter, points, metaData, sectionTitle) {


			var title = sectionTitle + " Values from " + filter.startDate + " to " + filter.endDate,
				chartConfig = getChartConfig(section, title);

			chartConfig.options.tooltip = configureTooltip(section, metaData);
			chartConfig.series = getSeriesConfig(section, points, metaData);
			chartConfig.xAxis = getXaxisConfig(section, filter);
			chartConfig.yAxis = getYaxisConfig(section, metaData);

			return chartConfig;
		};

		return service;
	});
});